<?php
/**
 * GSManager
 *
 * This is a mighty and platform independent software for administrating game servers of various kinds.
 * If you need help with installing or using this software, please visit our website at: www.gsmanager.de
 * If you have licensing enquiries e.g. related to commercial use, please contact us at: sales@gsmanager.de
 *
 * @copyright Greenfield Concept UG (haftungsbeschränkt)
 * @license GSManager EULA <https://www.gsmanager.de/eula.php>
 * @version 1.1.0
**/

namespace GSM\Plugins\KillMessages;

use GSM\Daemon\Core\Utils;

/**
 * killmessages
 *
 * The killmessages of the GSManager.
 *
 * @author Mirko911 <mirko911@gsmanager.de>
 */
class KillMessages extends Utils {
    /**
     * Firstblood
     *
     * Guid of the player who draw the first blood
     *
     * @var string Guid
     */
    private $firstblood = false;

    /**
     * Inits the plugin
     *
     * This function initiates the plugin. This means that it register commands
     * default values, and events. It's important that every plugin has this function
     * Otherwise the plugin exists but can't be used
     */
    public function initPlugin() {
        parent::initPlugin();

        foreach (array_keys($this->players) as $guid) {
            $this->playerInit($guid);
        }

        $this->config->setDefault('killmessages', 'enabled', true);
        $this->config->setDefault('killmessages', 'firstblood', true);
        $this->config->setDefault('killmessages', 'firstbloodmessage', '^4<PLAYER_NAME> ^5got first blood with ^4<WEAPON>^5!');
        $this->config->setDefault('killmessages', 'sequence', true);
        $this->config->setDefault('killmessages', 'sequencemessages', array());
        $this->config->setDefault('killmessages', 'timelimit', 3);
        $this->config->setDefault('killmessages', 'weapon', false);
        $this->config->setDefault('killmessages', 'weaponmessages', array());
    }

    /**
     * Function to enable this plugin
     */
    public function enable() {
        parent::enable();

        $this->events->register('playerKill', [$this, 'triggerKill']);
        $this->events->register('playerJoined', [$this, 'playerInit']);
        $this->events->register('nextMap', [$this, 'triggerNextMap']);
    }

    /**
     * Function to disable this plugin
     */
    public function disable() {
        parent::disable();

        $this->events->unregister('playerKill', [$this, 'triggerKill']);
        $this->events->unregister('nextMap', [$this, 'triggerNextMap']);
        $this->events->unregister('playerJoined', [$this, 'playerInit']);
    }

    /**
     * trigger a kill
     *
     * @param string $killer    GUID of the killer
     * @param string $victim    GUID of the victim
     * @param string $weapon    The weapon in question
     * @param int    $damage    amount of dmg done
     * @param string $bodypart  Hitzone
     * @return void
     */
    public function triggerKill($killer, $victim, $weapon, $damage, $bodypart) {

        //Firstblood
        $this->firstblood($killer, $victim, $weapon);
        $this->weaponkill($killer, $victim, $weapon);

        //Multikills
        if (time() - $this->players[$killer]->killmessages_lastkill <= $this->config->get('killmessages', 'timelimit')) {
            $this->players[$killer]->killmessages_multikills ++;
            $this->multiKill($killer, $this->players[$killer]->killmessages_multikills);
        } else {
            $this->players[$killer]->killmessages_multikills = 1;
        }
        $this->players[$killer]->killmessages_lastkill = time();
    }

    /**
     * trigger the nextmap
     */
    public function triggerNextMap() {
        $this->firstblood = false;
        foreach (array_keys($this->players) as $guid) {
            $this->playerInit($guid);
        }
    }

    /**
     * the spreemessage trigger
     *
     * @param string $message
     * @param string $killer
     * @param int $kills
     * @param int $deaths
     * @param string|bool $victim
     * @param string|bool weapon
     */
    private function spreemessage($message, $killer, $kills = 0, $deaths = 0, $victim = false, $weapon = false) {
        $search = array(
            '<PLAYER_NAME>',
            '<KILLS>',
            '<DEATHS>',
            '<KILLED_NAME>',
            '<WEAPON>'
        );

        if (is_array($weapon)) {
            if ($weapon[1] == "MOD_MELEE") {
                $weapon_info = $this->mod->getWeaponInfo('knife');
            } else {
                $weapon_info = $this->mod->getWeaponInfo(strtolower($weapon[0]));
            }
            $weapon = $weapon_info["name"];
        }

        $replace = array(
            $this->players[$killer]->getName(),
            $kills,
            $deaths,
            ($victim) ? $this->players[$victim]->getName() : '',
            ($weapon) ? $weapon : '',
        );
        $this->rcon->rconSay(str_replace($search, $replace, $message));
    }

    /**
     * Firtblood message
     *
     * @param string $killer
     * @param string $victim
     * @param string $weapon
     */
    private function firstblood($killer, $victim, $weapon) {
        if ($this->config->get('killmessages', 'firstblood') && !$this->firstblood) {
            $this->spreemessage($this->config->get('killmessages', 'firstbloodmessage'), $killer, 1, 0, $victim, $weapon);
        }
        $this->firstblood = true;
    }

    /**
     * multikill message
     *
     * @param string $killer
     * @param int $kills
     * @return void
     */
    private function multiKill($killer, $kills) {
        if (!$this->config->get('killmessages', 'sequence')) {
            return;
        }

        $messages = $this->config->get('killmessages', 'sequencemessages');
        if (array_key_exists($this->players[$killer]->killmessages_multikills, $messages)) {
            $this->spreemessage($messages[$this->players[$killer]->killmessages_multikills], $killer, $kills);
        }
    }

    public function weaponkill($killer, $victim, $weapon) {
        if (!$this->config->get('killmessages', 'weapon')) {
            return;
        }
        $messages = $this->config->get('killmessages', 'weaponmessages');
        if (array_key_exists(strtolower($weapon[0]), $messages)) {
            $this->spreemessage($messages[strtolower($weapon[0])], $victim, 0, 0, $killer, $weapon[0]);
        }
        if ($weapon[1] == "MOD_MELEE") {
            if (in_array("knife", $messages)) {
                $this->spreemessage($messages["knife"], $victim, 0, 0, $killer, "knife");
            }
        }
    }

    /**
     * playerInit for players
     *
     * @param string $guid
     */
    public function playerInit($guid) {
        $this->players[$guid]->killmessages_multikills = 0;
        $this->players[$guid]->killmessages_lastkill = 0;
    }
}
