<?php
/**
 * ManuAdminMod
 *
 * This is a powerful and platform independent software for administrating game servers of various kinds.
 * If you need help with installing or using this software, please visit our website at: www.ManuAdminMod.de
 * If you want to obtain additional permissions extending the license, contact us at: Webmaster@ManuAdminMod.de
 *
 * @copyright www.ManuAdminMod.de
 * @license http://www.creativecommons.org/licenses/by-nc-nd/4.0/ Creative Commons BY-NC-ND 4.0
 * @version 1.0.0-Beta+3
**/

namespace MAM\Plugins\TS3Poker;

use MAM\Daemon\Api\PluginHandler as PluginHandler;

/**
 * TS3Poker Plugin
 * Pokes one or more clients on a TeamSpeak3 server.
 *
 * config file:
 * {
 *     "ts3poker"          : {
 *         "enabled"       : true,
 *         "ip"            : "127.0.0.1",
 *         "port"          : 10011,
 *         "username"      : "test",
 *         "password"      : "b334ZBuH",
 *         "msgPrefix"     : "MAM here! ",
 *         "clients"       : "Ted,Angela"
 *     }
 * }
 * 
 * @author Yenz
 * @license http://creativecommons.org/licenses/by-nc-sa/4.0/ Creative Commons BY-NC-SA 4.0
 */
class TS3Poker extends PluginHandler {

	/**
     * Chars which aren't allowed in TS3 server query.
     * 
     * @var array $chars
     */
	private $chars = array("\\", "/", " ", "|", "\n", "\r", "\t");
	
	/**
     * Escape chars to translate the forbidden chars into TS3 server query format.
     * 
     * @var array $escape
     */
	private $escape = array("\\\\", "\\/", "\\s", "\\p", "\\n", "\\r", "\\t");
	
	/**
     * Constructor
     */
	public function __construct() {
		parent::__construct();
	}
	
    /**
     * Inits the plugin
     */
	public function initPlugin() {
		parent::initPlugin();
		
		$this->mod->setDefaultCV("ts3poker", "ip", "127.0.0.1");
		$this->mod->setDefaultCV("ts3poker", "port", 10011);
		$this->mod->setDefaultCV("ts3poker", "msgPrefix", "MAM here! ");
		
		$this->mod->registerCommand("ts3poke", false, "command", $this);
	}
	
    /**
     * Pokes one or more clients on a TeamSpeak3 server.
     * 
     * @param String $guid the player guid
     * @param array $parameters possible text message
     * @return void
     */
	public function command($guid, $parameters) {
		if (!$this->mod->getCV("ts3poker", "enabled")) {
			return;
		}
		$ip = $this->mod->getCV("ts3poker", "ip");
		$port = $this->mod->getCV("ts3poker", "port");
		$username = $this->mod->getCV("ts3poker", "username");
		$password = $this->mod->getCV("ts3poker", "password");
		$msg = $this->mod->getCV("ts3poker", "msgPrefix") . $this->players[$guid]->getName() . ": " . implode(" ", $parameters);
		
		$s = stream_socket_client("tcp://$ip:$port");
		
		stream_socket_sendto($s, "login $username $password\n");
		stream_socket_sendto($s, "use 1\n");
		$clients = $this->getClients($s);
		$pokedClients = array();
		foreach ($clients as $client) {
			stream_socket_sendto($s, "clientpoke clid=" . $client["clid"] . " msg=" . $this->TS3String($msg) . "\n");
			$pokedClients[] = $client["client_nickname"];
		}
		$this->logging->info("TS3Poker: poke " . implode(",", $pokedClients));
		$this->players[$guid]->say("count of poked guys: " . sizeof($pokedClients));
		stream_socket_sendto($s, "quit\n");
		stream_socket_shutdown($s, STREAM_SHUT_RDWR);
	}
	    
	/**
     * Searches for the TS clients.
     */
	private function getClients($s) {
		$clientNames = explode(",", $this->mod->getCV("ts3poker", "clients"));
		$clients = array();
		foreach ($clientNames as $clientName) {
			usleep(500000);
			$this->cleanInBuffer($s);
			stream_socket_sendto($s, "clientfind pattern=" . $this->TS3String($clientName) . "\n");
			usleep(500000);
			$ret = $this->decodeAnswer(stream_socket_recvfrom($s, 4096));
			foreach ($ret as $hit) {
				if ($this->PHPString($hit["client_nickname"]) == $clientName) {
					$clients[] = array("clid" => $hit["clid"], "client_nickname" => $clientName);
				}
			}
		}
		return $clients;
	}
	
	/**
	 * Cleans the TCP/stream input buffer
	 */
	private function cleanInBuffer($s) {
		do {
			$ret = stream_socket_recvfrom($s, 1024);
		} while ($ret != false);
	}
	
	/**
	 * Converts a string from PHP format into TS format.
	 */
	private function TS3String($string) {
		return str_replace($this->chars, $this->escape, $string);
	}
	
	/**
	 * Converts a string from TS format into PHP format.
	 */
	private function PHPString($string) {
		return str_replace($this->escape, $this->chars, $string);
	}
	
	/**
	 * Decodes an answer into an array structure to easily work with it.
	 */
	private function decodeAnswer($answer) {
		$answer = str_replace("\r", "", trim($answer)); //remove any unwanted whitespace
		$ret = array();
		$lines = explode("\n", $answer);
		for ($lineNr = 0; $lineNr < sizeof($lines); $lineNr++) {
			if (strpos($lines[$lineNr], "error id=") === 0) {
				// error handling? not yet!
			} else {
				$cells = explode("|", $lines[$lineNr]);
				for ($cellNr = 0; $cellNr < sizeof($cells); $cellNr++) {
					$parts = explode(" ", $cells[$cellNr]);
					foreach ($parts as $part) {
						if (strpos($part, "=") !== false) {
							$val = explode("=", $part, 2);
							$ret[$cellNr][$val[0]] = $this->PHPString($val[1]);
						}
					}
				}
			}
		}
		return $ret;
	}
}
