<?php
/**
 * GSManager
 *
 * This is a mighty and platform independent software for administrating game servers of various kinds.
 * If you need help with installing or using this software, please visit our website at: www.gsmanager.de
 * If you have licensing enquiries e.g. related to commercial use, please contact us at: sales@gsmanager.de
 *
 * @copyright Greenfield Concept UG (haftungsbeschränkt)
 * @license GSManager EULA <https://www.gsmanager.de/eula.php>
 * @version 1.1.0
 **/

namespace GSM\Daemon\Engines\Quake3;

use GSM\Daemon\Core\Utils;

/**
 * Game Commands
 *
 * @author Mirko911 <mirko911@gsmanager.de>
 */
class Commands extends Utils {

    /**
     * Returns a string if object is printed
     *
     * Notice: all plugins which where moved to core needs a __tostring function
     * to get it working with the permissions system. Therefore the return of __tostring must me the
     * namespace like deamon, punishment, permissions ...
     *
     * @see http://php.net/manual/en/language.oop5.magic.php#object.tostring
     * @return string
     */
    public function __toString() {
        return "daemon";
    }

    /**
     * Constructor
     *
     * Runs the plugin
     */
    public function __construct() {
        parent::__construct();
        $this->enable();
    }

    /**
     * Runs the plugin
     *
     * This function is called after all plugins were loaded and dependencies were checked.
     * In this function you can define things which you want to load, when all plugins are fine.
     *
     * If you know that you don't have any dependencies, you can remove this function.
     * Otherwise you should use it.
     *
     * In this function you should call functions like registerCommand, registerEvent, triggerEvent
     */
    public function enable() {
        parent::enable();

        $this->commands->register("set", false, false, $this);
        $this->commands->registerSubCommand("set", "password", false, "setPassword", $this);
        $this->commands->registerSubCommand("set", "oldschool", "~oldschool (0|1|off|on)~i", "setOldschool", $this);
        $this->commands->registerSubCommand("set", "killcam", "~killcam (0|1|off|on)~i", "setKillcam", $this);
        $this->commands->registerSubCommand("set", "gravity", "~gravity (\d+)?~i", "setGravity", $this);
        $this->commands->registerSubCommand("set", "knockback", "~knockback (\d+)?~i", "setKnockback", $this);
        $this->commands->registerSubCommand("set", "gravity", "~gravity (\d+)?~i", "setGravity", $this);
        $this->commands->registerSubCommand("set", "speed", "~speed (\d+)?~i", "setSpeed", $this);
        $this->commands->registerSubCommand("set", "spectator", "~spectator (0|1|2|off|team|all)~i", "setSpectator", $this);
        $this->commands->registerSubCommand("set", "voting", "~voting (0|1|off|on)~i", "setVoting", $this);
        $this->commands->registerSubCommand("set", "ff", "~ff (0|1|2|3|off|on|reflected|shared)~i", "setFF", $this);
        $this->commands->registerSubCommand("set", "killcam", "~killcam (0|1|off|on)~i", "setKillcam", $this);
        $this->commands->registerSubCommand("set", "hardcore", "~hardcore (0|1|off|on)~i", "setHardcore", $this);

        $this->commands->register("get", false, false, $this);
        $this->commands->registerSubCommand("get", "ff", false, "getFF", $this);
        $this->commands->registerSubCommand("get", "password", false, "getPassword", $this);
        $this->commands->registerSubCommand("get", "oldschool", false, "getOldschool", $this);
        $this->commands->registerSubCommand("get", "killcam", false, "getKillcam", $this);
        $this->commands->registerSubCommand("get", "gravity", false, "getGravity", $this);
        $this->commands->registerSubCommand("get", "knockback", false, "getKnockback", $this);
        $this->commands->registerSubCommand("get", "gravity", false, "getGravity", $this);
        $this->commands->registerSubCommand("get", "speed", false, "getSpeed", $this);
        $this->commands->registerSubCommand("get", "spectator", false, "getSpectator", $this);
        $this->commands->registerSubCommand("get", "voting", false, "getVoting", $this);
        $this->commands->registerSubCommand("get", "killcam", false, "getKillcam", $this);
        $this->commands->registerSubCommand("get", "hardcore", false, "getHardcore", $this);

        $this->commands->register("map", false, false, $this);
        $this->commands->registerSubCommand("map", "list", false, "mapList", $this);
        $this->commands->registerSubCommand("map", "change", '~change [\w. -]+~i', "mapChange", $this); /* @todo regex */
        $this->commands->registerSubCommand("map", "dev", '~dev [\w. -]+~i', "mapDev", $this); /* @todo regex */
        $this->commands->registerSubCommand("map", "set", '~set [\w. -]~i', "mapSet", $this);
        $this->commands->registerSubCommand("map", "restart", '~restart( (0|1))?~i', "mapRestart", $this);
        $this->commands->registerSubCommand("map", "get", false, "mapGet", $this);
        $this->commands->registerSubCommand("map", "rotate", false, "mapRotate", $this);


        $this->commands->register("gametype", false, false, $this);
        $this->commands->registerSubCommand("gametype", "list", false, "gametypeList", $this);
        $this->commands->registerSubCommand("gametype", "change", '~change \w+~i', "gametypeChange", $this);
        $this->commands->registerSubCommand("gametype", "set", '~set \w+~i', "gametypeSet", $this);
        $this->commands->registerSubCommand("gametype", "get", false, "gametypeGet", $this);



        $this->commands->registerSubCommand("quit", "srv", false, "killServer", $this);
        $this->commands->register("scream", '~^scream .+$~i', "scream", $this);
        $this->commands->register("status", false, "status", $this);
        if ($this->registry->iceops) {
            $this->commands->register("yell", '~^yell .+$~i', "yell", $this);
        }
    }

    /**
     * Change the current map
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters part of mapname
     */
    public function mapChange($guid, $parameters) {
        //no case sensitivity
        $parameters = array_map("strtolower", $parameters);

        if (count($parameters) > 1 && in_array($parameters[count($parameters) - 1], array("sd", "war", "dm", "sab", "koth", "dom", "ctf", "htf", "twar", "tdm", "hq", "gtnw", "oneflag", "dd", "arena"))) {
            $gametype = array_pop($parameters);
        } else {
            $gametype = $this->mod->getCurrentGametype();
        }

        $map = $this->mod->findMap(implode(" ", $parameters));

        if (!$map) {
            $this->players[$guid]->say($this->language->get("daemon.mapNotFound", array("<MAP>"), array(implode(" ", $parameters))));
            return;
        }

        $this->rcon->rconSay($this->language->get("daemon.changeMap", array("<MAP>", "<GAMETYPE>"), array($this->mod->getLongMapName($map), $this->mod->getLongGametype($gametype))));
        if (!$this->rcon->rconChangeMap($map, $gametype)) {
            $this->rcon->rconSay($this->language->get("daemon.mapNotFound", array("<MAP>"), array(implode(" ", $parameters))));
        }
    }

    /**
     * Restart the current map
     *
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     * @return void
     */
    public function mapRestart($guid, $parameters) {
        if (empty($parameters[0])) {
            $this->rcon->rconSay($this->language->get("daemon.mapRestart"));
            $this->rcon->rconMapRestart(false);
        } else {
            $parameters[0] = str_replace(array("slow", "fast"), array(false, true), strtolower($parameters[0]));
            $this->rcon->rconSay($this->language->get("daemon.mapFastRestart"));
            $this->rcon->rconMapRestart($parameters[0]);
        }
    }

    /**
     * Kill the GameServer
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function killServer($guid, $parameters) {
        $this->rcon->rconKillServer((array_key_exists(0, $parameters)) ? $parameters[0] : 1, (array_key_exists(1, $parameters)) ? $parameters[1] : 3);
    }

    /**
     * Loads the next map in the rotation
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function mapRotate($guid, $parameters) {
        $this->rcon->rconSay($this->language->get("daemon.loadingNextMap"));
        $this->rcon->rconMapRotate();
    }

    /**
     * gametype changes the gametype reload the current map with the selected gametype directly
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function gametypeChange($guid, $parameters) {
        if (!$this->permissions->isPlayerAllowed($guid, 'daemon.gametype.change.' . $parameters[0])) {
            $this->players[$guid]->say($this->language->get('permissions.notenough'));
            return false;
        }
        $this->rcon->rconSay($this->language->get("daemon.changeGametype", array("<GAMETYPE>"), array($this->mod->getLongGametype($parameters[0]))));
        $this->rcon->rconSetGametype($parameters[0]);
        $this->rcon->rconMapRestart(false);
    }

    /**
     * gametype changes the gametype reload the current map with the selected gametype directly
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function gametypeList($guid, $parameters) {
        $this->players[$guid]->say($this->language->get("daemon.gametypelist", array("<GAMETYPES>"), array($this->mod->getGametypes(","))));
    }

    /**
     * returns a message with the next map to the console
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function mapGet($guid, $parameters) {
        $next = $this->rcon->rconGetNextMap();
        $map = $this->mod->getLongMapName($next["map"]);
        $gt = $this->mod->getLongGametype($next["gametype"]);
        $this->rcon->rconSay($this->language->get("daemon.nextmap", array("<GAMETYPE>", "<MAP>"), array($gt, $map)));
    }

    /**
     * set next gametype
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function gametypeSet($guid, $parameters) {
        $gamemode = $this->getShortGamemode($parameters[0]);
        $next = $this->rcon->rconGetNextMap();
        $map = $this->mod->getLongMapName($next["map"]);
        if ($gamemode === false) {
            $this->players[$guid]->say($this->language->get("daemon.gamemodeNotExist", array("<GAMETYPE>"), array($parameters[0])));
            return;
        }

        if ($this->rcon->rconSetNextMap($next["map"], $gamemode, 2)) {
            $this->rcon->rconSay($this->language->get("daemon.setNextMap", array("<MAP>", "<GAMETYPE>"), array($this->mod->getLongMapName($map), $this->mod->getLongGametype($gamemode))));
        } else {
            $this->players[$guid]->say($this->language->get("daemon.error"));
        }
    }

    /**
     * returns a message with the next map to the console
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function gametypeGet($guid, $parameters) {
        $next = $this->rcon->rconGetNextMap();
        $gt = $this->mod->getLongGametype($next["gametype"]);
        $this->rcon->rconSay($this->language->get("daemon.nextgametype", array("<GAMETYPE>"), array($gt)));
    }

    /**
     * sets a password on the gameserver
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function setPassword($guid, $parameters) {
        if (!empty($parameters)) {
            $this->rcon->rconSay($this->language->get("daemon.passwordSet"));
            $this->rcon->rconSetDvar("g_password", $parameters[0]);
        } else {
            $this->rcon->rconSay($this->language->get("daemon.passwordRemoved"));
            $this->rcon->rconSetDvar("g_password", "");
        }
    }

    public function getPassword($guid, $parameters) {
        $dvar = $this->rcon->rconGetDvar("g_password");
        $this->players[$guid]->say($this->language->get("daemon.get", array("<VALUE>", "<MODE>"), array("Password", "'" . $dvar . "'")));
    }

    /**
     * turn hardcore on/off on the server for the next map/round
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function setHardcore($guid, $parameters) {
        $parameters[0] = str_replace(array("off", "on"), array(0, 1), strtolower($parameters[0]));
        $mode = array($this->language->get("daemon.disabled"), $this->language->get("daemon.enabled"));
        $this->rcon->rconSay($this->language->get("daemon.hardcoreChange", array("<MODE>"), array($mode[$parameters[0]])));
        $this->rcon->rconSetDvar("scr_hardcore", $parameters[0]);
    }

    public function getHardcore($guid, $parameters) {
        $dvar = $this->rcon->rconGetDvar("scr_hardcore");
        $mode = array($this->language->get("daemon.disabled"), $this->language->get("daemon.enabled"));
        $this->players[$guid]->say($this->language->get("daemon.get", array("<VALUE>", "<MODE>"), array("Hardcore", $mode[$dvar])));
    }

    /**
     * enable or disabled ingame vote
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function setVoting($guid, $parameters) {
        $parameters[0] = str_replace(array("off", "on"), array(0, 1), strtolower($parameters[0]));
        $mode = array($this->language->get("daemon.disabled"), $this->language->get("daemon.enabled"));
        $this->rcon->rconSay($this->language->get("daemon.votingChange", array("<MODE>"), array($mode[$parameters[0]])));
        $this->rcon->rconSetDvar("ui_allowvote", $parameters[0]);
    }

    public function getVoting($guid, $parameters) {
        $dvar = $this->rcon->rconGetDvar("ui_allowvote");
        $mode = array($this->language->get("daemon.disabled"), $this->language->get("daemon.enabled"));
        $this->players[$guid]->say($this->language->get("daemon.get", array("<VALUE>", "<MODE>"), array("Voting", $mode[$dvar])));
    }

    /**
     * turn oldschool on/off on the server for the next map/round
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function setOldschool($guid, $parameters) {
        $parameters[0] = str_replace(array("off", "on"), array(0, 1), strtolower($parameters[0]));
        $mode = array($this->language->get("daemon.disabled"), $this->language->get("daemon.enabled"));
        $this->rcon->rconSay($this->language->get("daemon.oldschoolChange", array("<MODE>"), array($mode[$parameters[0]])));
        $this->rcon->rconSetDvar("scr_oldschool", $parameters[0]);
    }

    public function getOldschool($guid, $parameters) {
        $dvar = $this->rcon->rconGetDvar("scr_oldschool");
        $mode = array($this->language->get("daemon.disabled"), $this->language->get("daemon.enabled"));
        $this->players[$guid]->say($this->language->get("daemon.get", array("<VALUE>", "<MODE>"), array("Oldschool", $mode[$dvar])));
    }

    /**
     * turn killcam on/off on the server for the next map/round
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function setKillcam($guid, $parameters) {
        $parameters[0] = str_replace(array("off", "on"), array(0, 1), strtolower($parameters[0]));
        $mode = array($this->language->get("daemon.disabled"), $this->language->get("daemon.enabled"));
        $this->rcon->rconSay($this->language->get("daemon.killcamChange", array("<MODE>"), array($mode[$parameters[0]])));
        $this->rcon->rconSetDvar("scr_game_allowkillcam", $parameters[0]);
    }

    public function getKillcam($guid, $parameters) {
        $dvar = $this->rcon->rconGetDvar("scr_game_allowkillcam");
        $mode = array($this->language->get("daemon.disabled"), $this->language->get("daemon.enabled"));
        $this->players[$guid]->say($this->language->get("daemon.get", array("<VALUE>", "<MODE>"), array("Killcam", $mode[$dvar])));
    }

    /**
     * sets the gravity to x on the server
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function setGravity($guid, $parameters) {
        if (!empty($parameters[0])) {
            $VALUE = $parameters[0];
        } else {
            $VALUE = 800;
        }
        $this->rcon->rconSay($this->language->get("daemon.gravityChange", array("<VALUE>"), array($VALUE)));
        $this->rcon->rconSetDvar("g_gravity", $VALUE);
    }

    public function getGravity($guid, $parameters) {
        $dvar = $this->rcon->rconGetDvar("g_gravity");
        $this->players[$guid]->say($this->language->get("daemon.get", array("<VALUE>", "<MODE>"), array("Gravity", $dvar)));
    }

    /**
     * sets the knockback to x on the server
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function setKnockback($guid, $parameters) {
        if (!empty($parameters[0])) {
            $VALUE = $parameters[0];
        } else {
            $VALUE = 1000;
        }
        $this->rcon->rconSay($this->language->get("daemon.knockbackChange", array("<VALUE>"), array($VALUE)));
        $this->rcon->rconSetDvar("g_knockback", $VALUE);
    }

    public function getKnockback($guid, $parameters) {
        $dvar = $this->rcon->rconGetDvar("g_knockback");
        $this->players[$guid]->say($this->language->get("daemon.get", array("<VALUE>", "<MODE>"), array("Knockback", $dvar)));
    }

    /**
     * sets the speed to x on the server
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function setSpeed($guid, $parameters) {
        if (!empty($parameters[0])) {
            $VALUE = $parameters[0];
        } else {
            $VALUE = 190;
        }
        $this->rcon->rconSay($this->language->get("daemon.speedChange", array("<VALUE>"), array($VALUE)));
        $this->rcon->rconSetDvar("g_speed", $VALUE);
    }

    public function getSpeed($guid, $parameters) {
        $dvar = $this->rcon->rconGetDvar("g_speed");
        $this->players[$guid]->say($this->language->get("daemon.get", array("<VALUE>", "<MODE>"), array("Speed", $dvar)));
    }

    /**
     * sets the friendly fire to x
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function setff($guid, $parameters) {
        $parameters[0] = str_replace(array("shared", "reflected", "on", "off"), array(3, 2, 1, 0), strtolower($parameters[0]));

        $mode = array(
            $this->language->get("daemon.disabled"),
            $this->language->get("daemon.enabled"),
            $this->language->get("daemon.reflected"),
            $this->language->get("daemon.shared"),
        );
        $this->rcon->rconSay($this->language->get("daemon.setFriendlyFire", array("<MODE>"), array($mode[$parameters[0]])));
        $this->rcon->rconSetDvar("scr_team_fftype", $parameters[0]);
    }

    /**
     * retuns a message that shows the friendly fire
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function getFF($guid, $parameters) {
        $ff = $this->rcon->rconGetDvar("scr_team_fftype");
        $mode = array(
            $this->language->get("daemon.disabled"),
            $this->language->get("daemon.enabled"),
            $this->language->get("daemon.reflected"),
            $this->language->get("daemon.shared"),
        );
        $this->rcon->rconSay($this->language->get("daemon.friendlyFire", array("<MODE>"), array($mode[$ff])));
    }

    /**
     * Set Spectatormode
     * @param String $guid Guid of executing Player
     * @param String $parameters Additional parameters Parameter of this command
     */
    public function setSpectator($guid, $parameters) {
        $parameters[0] = str_replace(array("all", "team", "off"), array(2, 1, 0), strtolower($parameters[0]));
        $mode = array($this->language->get("daemon.disabled"), "Team", "All");
        $this->rcon->rconSay($this->language->get("daemon.spectatorChange", array("<MODE>"), array($mode[$parameters[0]])));
        $this->rcon->rconSetDvar("scr_game_spectatetype", $parameters[0]);
    }

    public function getSpectator($guid, $parameters) {
        $dvar = $this->rcon->rconGetDvar("scr_game_spectatetype");
        $mode = array($this->language->get("daemon.disabled"), "Team", "All");
        $this->players[$guid]->say($this->language->get("daemon.get", array("<VALUE>", "<MODE>"), array("Spectator", $mode[$dvar])));
    }

    /**
     * start a map with enabled cheats
     *
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     * @return void
     */
    public function mapDev($guid, $parameters) {
        //no case sensitivity
        $parameters = array_map("strtolower", $parameters);

        $map = $this->mod->findMap(implode(" ", $parameters));

        if (!$map) {
            $this->players[$guid]->say($this->language->get("daemon.mapNotFound", array("<MAP>"), array(implode(" ", $parameters))));
            return false;
        }

        $this->rcon->rconSay($this->language->get("daemon.changeMapDevMap", array("<MAP>"), array($this->mod->getLongMapName($map))));
        if (!$this->rcon->rconDevMap($map)) {
            $this->rcon->rconSay($this->language->get("daemon.mapNotFound", array("<MAP>"), array(implode(" ", $parameters))));
        }
    }

    /**
     * shows the usable maps
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function maplist($guid, $parameters) {
        $this->rcon->rconSay($this->mod->getMaps("; "));
    }

    /**
     * sets the next map to x
     *
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     * @return void
     */
    public function mapSet($guid, $parameters) {
        $parameters = array_map("strtolower", $parameters);

        //war,dm,sab,koth,sd,dom,gtnw,oneflag,ctf,dd,arena

        if (count($parameters) > 1 && in_array($parameters[count($parameters) - 1], array("sd", "war", "dm", "sab", "koth", "dom", "ctf", "htf", "twar", "tdm", "hq", "gtnw", "oneflag", "dd", "arena"))) {
            $gametype = array_pop($parameters);
        } else {
            $gametype = $this->mod->getCurrentGametype();
        }

        $map = $this->mod->findMap(implode(" ", $parameters));

        if (!$map) {
            $this->players[$guid]->say($this->language->get("daemon.mapNotFound", array("<MAP>"), array(implode(" ", $parameters))));
            return false;
        }

        $this->rcon->rconSetNextMap($map, $gametype, true);

        $this->rcon->rconSay($this->language->get("daemon.setNextMap", array("<MAP>", "<GAMETYPE>"), array($this->mod->getLongMapName($map), $this->mod->getLongGametype($gametype))));
    }

    /**
     * status about the currently running adminmod
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function status($guid, $parameters) {
        $parsed = $this->registry->parser->getParsedLogLines();
        $up_time = \GSM\Daemon\Libraries\Helper\Helper::convertSeconds($this->registry->parser->getLastParsedTimeStamp()); ;
        $bytes = \GSM\Daemon\Libraries\Helper\Helper::size($this->registry->parser->getParsedBytes());
        $time = \GSM\Daemon\Libraries\Helper\Helper::convertSeconds(time() - $this->registry->startuptime, "%dd,%hh,%imin");
        $ram = \GSM\Daemon\Libraries\Helper\Helper::size(memory_get_usage());

        $msg = $this->language->get("daemon.status", array("<PARSEDLINES>", "<PARSEDBYTES>", "<UPTIME>", "<MEMUSAGE>", "<SERVERUPTIME>"), array($parsed, $bytes, $time, $ram, $up_time));
        $this->rcon->rconSay($msg);
    }

    /**
     * Screams a message 9 times on the server
     *
     * @param String $guid Guid of executing player
     * @param String $params Additional parameters
     */
    public function scream($guid, $params) {
        $msg = implode(" ", $params);
        $this->mod->findReason($msg);

        $msg = (strlen($msg) > 100) ? substr($msg, 0, 100) : $msg;
        for ($i = 0; $i <= 9; $i ++) {
            $this->rcon->rconSay("^$i$msg");
        }
    }

    /**
     * yell to all player
     * @param String $guid Guid of executing player
     * @param String $parameters Additional parameters
     */
    public function yell($guid, $parameters) {
        $search = strtolower(array_shift($parameters));
        $msg = implode(" ", $parameters);
        $this->mod->findReason($msg);

        if ($search == 'all' || $search == '*') {
            $this->rcon->rcon('screensay ' . $msg);
        } elseif ($player = $this->mod->findPlayerGuid($search) != false) {
            $this->rcon->rcon('screentell ' . $this->players[$player]->getPID() . ' ' . $msg);
        } else {
            $this->players[$guid]->say($this->language->get("daemon.playernotfound", "<SEARCH>", $search));
            /** @todo Printhelp needed here? */
            $this->mod->printHelp((string) $this, 'yell', $guid);
        }
    }
}
