<?php
/**
 * Ban IP Range Plugin
 * This file contains the class for the Ban Ip Range Plugin.
 * 
 * Configuration:
 *		Edit banend ranges in config/plugins/banned_ip_ranges.txt
 *		Example:
 *			; this is a comment an will be ignored
 *			147.16.0.0 - 147.16.255.255
 *			151.65.13.0-151.65.13.127
 *			1.0.0.0- 20.0.0.0
 *		After editing just execute !readconfig to apply changes.
 * 
 * @author Yenz
 * @license Creative Commons BY-NC-SA 3.0 (http://www.creativecommons.org/licenses/by-nc-sa/3.0/)
 * @version 1.0
 */

$banIpRange = new banIpRange();

$mod->registerEvent("playerJoined", "playerJoined", $banIpRange);

/**
 * The Ban IP Range class controls ips from joining players and checks if one is within a banned range.
 * In that case the player will be kicked.
 */
class banIpRange
{
	/**
	 * @var array every entry is an array with two elements, the first and last ip of a range
	 */
	private $ranges = array();
	
	/**
	 * Instantiates the Ban IP Range Plugin
	 */
	public function __construct()
	{
		$this->readConfig();
	}
	
	/**
	 * Reads the banned ranges from file and creates the ranges array.
	 * @global mod $mod
	 */
	public function readConfig()
	{
		global $mod;
		$this->ranges = array();
		
		$cfgFile = $mod->getConfigDir() . "/plugins/banned_ip_ranges.txt";
		if (!file_exists($cfgFile))
		{
			fclose(fopen($cfgFile, "a"));
		}
		$file = file($cfgFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
		for ($lineNum = 1; $lineNum <= count($file); $lineNum++)
		{
			$line = trim($file[$lineNum - 1]);
			if ($line != "" && substr($line, 0, 1) != ";")
			{
				$p = explode("-", $line);
				if (count($p) != 2)
				{
					$this->printSyntaxError($lineNum);
					continue;
				}

				$ip_a = $this->ipString2Int(trim($p[0]));
				$ip_b = $this->ipString2Int(trim($p[1]));
				if ($ip_a === false || $ip_b === false || $ip_a > $ip_b)
				{
					$this->printSyntaxError($lineNum);
					continue;
				}
				
				$this->ranges[] = array($ip_a, $ip_b);
			}
		}
	}
	
	/**
	 * Will be called when a player joins and will kick him, if his ip is within a banned range.
	 * @global mod $mod
	 * @global player $players
	 * @param string $guid
	 */
	public function playerJoined($guid)
	{
		global $mod, $players;
		$rconPlayerList = $mod->rconPlayerList();
		foreach($rconPlayerList as $rconPlayer)
		{
            if ($rconPlayer["guid"] == $guid)
			{
				$ip_p = $this->ipString2Int($rconPlayer["ip"]);
				if ($this->isIpInRange($ip_p))
				{
					$players[$guid]->kick("ip range ban");
					$logging->write(MOD_NOTICE, "baniprange plugin kicked player with ip " . $rconPlayer["ip"]);
				}
				break;
            }
        }
	}
	
	/**
	 * Checks if the given ip is within a banned range.
	 * @param int $ip ip in integer format
	 */
	private function isIpInRange($ip)
	{
		foreach ($this->ranges as $range)
		{
			if ($ip >= $range[0] && $ip <= $range[1])
			{
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Print an error to the log if the config file contains syntax error.
	 * @param int $lineNum the line number which contains the error
	 */
	private function printSyntaxError($lineNum)
	{
		global $mod, $logging;
		$logging->write(MOD_WARNING, "syntax error in " . $mod->getConfigDir() . "/plugins/banned_ip_ranges.txt in line $lineNum");
	}
	
	/**
	 * Converts an ip from decimal representation to integer.
	 * @param string $ipString (e.g. 168.15.19.4)
	 */
	private function ipString2Int($ipString)
	{
		$p = explode(".", $ipString);
		
		if (count($p) != 4
			|| $p[0] < 0 || $p[0] > 255
			|| $p[1] < 0 || $p[1] > 255
			|| $p[2] < 0 || $p[2] > 255
			|| $p[3] < 0 || $p[3] > 255)
		{
			return false;
		}
		
		return $p[0] * 256 * 256 * 256 + $p[1] * 256 * 256 + $p[2] * 256 + $p[3];
	}
}
