<?php
/**
 * Ban IP Plugin
 * This file contains the class for the Ban Ip Plugin and manage the mod to call
 * the user functions !banip, !tempbanip and !unbanip.
 * 
 * Commands:
 *      !banip player [reason]
 *          Kick the player and list its ip as blacklisted.
 *      !tempbanip player duration [reason]
 *          Kick the player and list its ip for "duration" minutes.
 *      !unbanip id|ip|nick
 *          Mark the entry not longer as blacklisted. You can use the id of the
 *          entry in the banip.db database or the specific ip or search after
 *          the nickname of the player it has got last time.
 * 
 * Events:
 *      PlayerJoined
 *          A player who is joining the server will be kicked if its ip is blacklisted.
 * 
 * Old tempban entries will be deleted automatically.
 * 
 * Config:
 *      [banip]
 *      enabled = 1
 *      usedefaulttempbanduration = 1
 *      defaulttempbanduration = 30
 *      cleanupeveryhour = 48
 * 
 * @author silva202
 * @license Creative Commons BY-NC-SA 3.0 (http://www.creativecommons.org/licenses/by-nc-sa/3.0/)
 * @version 2.0
 */

$banIp = new banIp();

// events and commands
$mod->registerEvent("playerJoined", "triggerPlayerJoined", $banIp);
$mod->registerCommand("banip", false, "commandBanIp", $banIp);
$mod->registerCommand("tempbanip", false, "commandTempBanIp", $banIp);
$mod->registerCommand("unbanip", false, "commandUnbanIp", $banIp);

// default CVars
$mod->setDefaultCV("banip", "enabled", 0);
$mod->setDefaultCV("banip", "usedefaulttempbanduration", 1);
$mod->setDefaultCV("banip", "defaulttempbanduration", 30);
$mod->setDefaultCV("banip", "cleanupeveryhour", 48);


/**
 * The Ban IP Plugin class offers the commands for banning, tempbanning and
 * unbanning players on the basis ot its ip address. There fore a SQLite 3
 * database is used, so you need PHP >= 5.3.0 and the sqlite3 extension.
 */
class banIp {
    /**
     * @var SQLite3 database
     */
    private $db;
    
    /**
     * @var int timestamp of last cleanup
     */
    private $lastCleanup;

    /**
     * Instantiates the Ban IP Plugin.
     * A SQLite3 database will be created, if it doesn't exists.
     * @global mod $mod
     */
    public function __construct() {
        global $mod;

        // open SQLite3 database (will be created if it doesn't exist)
        $this->db = new SQLite3($mod->getConfigDir() . "/plugins/banip.db");

        // create table if it doesn't exist
        $createTable = "CREATE TABLE IF NOT EXISTS banip (
            id INTEGER PRIMARY KEY,
            ip TEXT NOT NULL UNIQUE,
            nick TEXT NOT NULL,
            time INTEGER NOT NULL,
            kicker TEXT NOT NULL,
            reason TEXT,
            duration INTEGER NOT NULL)"; // duration in minutes
        $this->db->exec($createTable);
        
        $this->lastCleanup = 0;
    }
    
    /**
     * Check a joined player if his ip is blacklisted.
     * @global mod $mod
     * @global player $players
     * @global log $logging
     * @param string $guid 
     */
    public function triggerPlayerJoined($guid) {
        global $mod, $players, $logging;
        
        if ($mod->getCV("banip", "enabled") != 1) {
            return;
        }
        
        $this->cleanup();
        
        $rconPlayerList = $mod->rconPlayerList();
        
        foreach($rconPlayerList as $rconPlayer) {
            if ($rconPlayer["guid"] == $guid && $this->isBanned($rconPlayer["ip"])) {
                // update nick
                $stmt = $this->db->prepare("UPDATE banip SET nick = :nick WHERE ip = '" . $rconPlayer["ip"] . "'");
                $stmt->bindValue(":nick", $players[$guid]->getName());
                $stmt->execute();
                
                // get ban details
                $row = $this->db->querySingle("SELECT * FROM banip WHERE ip = '" . $rconPlayer["ip"] . "'", true);
                
                $logMsg = "BanIP-Plugin: Kicking banned player: ip: " . $row["ip"] . ", nick: " . $row["nick"];
                
                if (empty($reason)) {
                    $players[$guid]->kick();
                    
                } else {
                    $players[$guid]->kick($row["reason"]);
                    $logMsg .= ", reason: " . $row["reason"];
                }
                
                $logging->write(MOD_NOTICE, $logMsg);
            }
        }
    }
    
    /**
     * Command !banip player (reason)
     * @global mod $mod
     * @global player $players
     * @param string $guid
     * @param array $parameters 
     */
    public function commandBanIp($kickerGuid, $parameters) {
        global $mod, $players;
        
        if ($mod->getCV("banip", "enabled") != 1) {
            return;
        }
        
        $this->cleanup();
        
        if (count($parameters) < 1) {
            $players[$kickerGuid]->say("Usage: !banip player [reason]");
            return false;
        }
        
        $playerToBan = $parameters[0];
        
        if (count($parameters) == 1) {
            $reason = false;
        } else {
            $reason = "";
            for ($n = 1; $n < count($parameters); $n++) {
                $reason .= $parameters[$n] . " ";
            }
        }
        
        $this->banIp($kickerGuid, $playerToBan, 0, trim($reason));
    }
    
    /**
     * Command !tempbanip player duration (reason)
     * @global mod $mod
     * @global player $players
     * @param string $guid
     * @param array $parameters 
     */
    public function commandTempBanIp($kickerGuid, $parameters) {
        global $mod, $players;
        
        if ($mod->getCV("banip", "enabled") != 1) {
            return;
        }
        
        $this->cleanup();
        
        $parametersError = false;
        
        // no parameters given
        if (count($parameters) < 1) {
            $parametersError = true;
        }
        
        // only one parameters given
        if (count($parameters) < 2) {
            if ($mod->getCV("banip", "usedefaulttempbanduration") == 1) {
                // set default tempban duration as second parameter
                $parameters[1] = $mod->getCV("banip", "defaulttempbanduration");
                
            } else {
                $parametersError = true;
            }
        }
        
        // check if duration is a numeric parameter
        if (!is_numeric($parameters[1])) {
            $parametersError = true;
        }
        
        // abort if parameters contain errors
        if ($parametersError) {
            if ($mod->getCV("banip", "usedefaulttempbanduration") == 1) {
                $players[$kickerGuid]->say("Usage: !tempbanip player [duration reason]");
                
            } else {
                $players[$kickerGuid]->say("Usage: !tempbanip player duration [reason]");
            }
            
            return false;
        }
        
        $playerToBan = $parameters[0];
        $duration = $parameters[1];
        
        if (count($parameters) == 2) {
            $reason = false;
        } else {
            $reason = "";
            for ($n = 2; $n < count($parameters); $n++) {
                $reason .= $parameters[$n] . " ";
            }
        }
        
        $this->banIp($kickerGuid, $playerToBan, $duration, trim($reason));
    }
    
    /**
     * Write the playerToBan's ip into the database and kick the player.
     * @global mod $mod
     * @global player $players
     * @param string $kickerGuid
     * @param string $playerToBan
     * @param int $duration
     * @param string|boolean $reason
     * @return boolean false if $playerToBan couldn't be found
     */
    private function banIp($kickerGuid, $playerToBan, $duration = 0, $reason = false) {
        global $mod, $players;
        
        $playerToBanGuid = $mod->findPlayerGuid($playerToBan);

        if ($playerToBanGuid == false) {
            $players[$kickerGuid]->say($mod->getLngString("playerNotFound", array("<SEARCH>"), array($playerToBan)));
            return false;
        }

        $rconPlayerList = $mod->rconPlayerList();
        foreach($rconPlayerList as $rconPlayer) {
            if ($rconPlayer["guid"] == $playerToBanGuid) {
                $stmt = $this->db->prepare("INSERT INTO banip (ip, nick, time, kicker, reason, duration) VALUES(:ip, :nick, :time, :kicker, :reason, :duration)");
                
                if ($reason != false) {
                    $stmt->bindValue(":reason", $reason, SQLITE3_TEXT);

                } else {
                    $stmt->bindValue(":reason", null, SQLITE3_NULL);
                    
                }
                
                $stmt->bindValue(":ip", $rconPlayer["ip"], SQLITE3_TEXT);
                $stmt->bindValue(":nick", $players[$playerToBanGuid]->getName(), SQLITE3_TEXT);
                $stmt->bindValue(":time", time(), SQLITE3_INTEGER);
                $stmt->bindValue(":kicker", $kickerGuid, SQLITE3_TEXT);
                $stmt->bindValue(":duration", $duration, SQLITE3_INTEGER);

                $stmt->execute();
                
                $players[$playerToBanGuid]->kick($reason, $kickerGuid);

            }
        }
    }
    
    /**
     * Check if the ip is blacklisted.
     * @param string $ip
     * @return boolean true, if ip is blacklisted
     */
    private function isBanned($ip) {
        $row = $this->db->querySingle("SELECT * FROM banip WHERE ip = '$ip'", true);
        
        if (empty($row)) {
            // no data for this ip
            return false;
            
        } else if ($row["duration"] == 0) {
            // no duration for this ip
            return true;
            
        } else if ($row["time"] + $row["duration"] * 60 > time()) {
            // duration is not over
            return true;
            
        } else {
            // old tempban
            $this->db->exec("DELETE FROM banip WHERE ip = '$ip'");
            return false;
            
        }
    }
    
    /**
     * Command !unbanip player.
     * @global mod $mod
     * @global player $players
     * @param string $guid
     * @param array $parameters 
     */
    public function commandUnbanIp($guid, $parameters) {
        global $mod, $players;
        
        if ($mod->getCV("banip", "enabled") != 1) {
            return;
        }
        
        $this->cleanup();
        
        $playerToUnban = array();
        
        if (count($parameters) < 1) {
            $players[$guid]->say("Usage: !unbanip id|ip|part_of_nick");
            return false;
        }
        
        $parameter = implode(" ", $parameters);
        
        if (is_numeric($parameter)) {
            // parameter is an id
            $playerToUnban = $this->db->querySingle("SELECT * FROM banip WHERE id = '$parameter'", true);
            
            if (empty($playerToUnban)) {
                $players[$guid]->say("No entry was found for id: $parameter");
            }
            
        } else if ($this->isIp($parameter)) {
            // parameter is an ip
            $stmt = $this->db->prepare("SELECT * FROM banip WHERE ip = :ip");
            $stmt->bindValue(":ip", $parameter, SQLITE3_TEXT);
            $playerToUnban = $stmt->execute()->fetchArray();
            
            if (empty($playerToUnban)) {
                $players[$guid]->say("No entry was found for ip: $parameter");
            }
            
        } else {
            // parameter is a part of a nick
            $stmt = $this->db->prepare("SELECT * FROM banip WHERE nick LIKE :nick");
            $stmt->bindValue(":nick", "%" . $parameter . "%", SQLITE3_TEXT);
            $result = $stmt->execute();
            
            $found = array();
            while ($row = $result->fetchArray()) {
                $found[] = $row;
            }
            
            
            if (count($found) == 0) {
                // no one found
                $players[$guid]->say("No entry was found for nick: $parameter");
                
            } else if (count($found) == 1) {
                // exactly one entry found, delete
                $playerToUnban = $found[0];
                
            } else if (count($found) <= 10) {
                // some entries found, list them via pm
                $players[$guid]->say("Some entries were found, use !unbanip id:");
                foreach ($found as $row) {
                    $players[$guid]->say($row["id"] . " | " . $row["ip"] . " | " . $row["nick"]);
                }
                
            } else {
                // too many entries found, send pm
                $players[$guid]->say("Too many entries (" . count($found) . ") found, please be more specific or use ip instead.");
            }
        }
        
        if (!empty($playerToUnban)) {
            $this->db->exec("DELETE FROM banip WHERE id = '" . $playerToUnban["id"] . "'");
            $players[$guid]->say("banip removed: " . $playerToUnban["id"] . " | " . $playerToUnban["ip"] . " | " . $playerToUnban["nick"] . ")");
        }
    }
    
    /**
     * Check if the given string is a valid ip.
     * @param string $ip
     * @return boolean true if string $ip is an ip, otherwise false
     */
    private function isIp($ip) {
        $parts = explode(".", $ip);
        
        if (count($parts) != 4) {
            return false;
        }
        
        foreach ($parts as $num) {
            if ($num < 0 || $num > 255) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Cleanup routine that deletes old tempban entries from database.
     * A cleanup will only run if last cleanup was before x hours (x can be
     * set in config under [banip]cleanupeveryhour).
     * @global mod $mod 
     * @global log $logging
     */
    private function cleanup() {
        global $mod, $logging;
        
        if ($this->lastCleanup + $mod->getCV("banip", "cleanupeveryhour") * 3600 < time()) {
            $this->db->exec("DELETE FROM banip WHERE time + duration * 60 < " . time() . " AND duration <> 0");
            
            $logging->write(MOD_NOTICE, "Ban IP Plugin: Cleanup deleted " . $this->db->changes() . " old tempban entries.");
            
            $this->lastCleanup = time();
        }
    }

}
